#include "GUI/MiniGUI.h"
#include "Turing/Turing.h"
#include <iostream>
#include <fstream>
using namespace std;

CONSOLE_HANDLER("Turing Demo") {
    while (true) {
        string input = getLine("Input: ");

        Turing::Program p(ifstream("res/AnBn.tm", ios::binary));
        if (!p.isValid()) {
            for (size_t i = 0; i < p.numLines(); i++) {
                cout << p.line(i) << endl;
                string error = p.errorAtLine(i);
                if (error != "") {
                    cerr << "  " << error << endl;
                }
            }
        } else {
            Turing::Interpreter interpreter(p, { input.begin(), input.end() });

            while (true) {
                for (size_t i = 0; i < 100; i++) {
                    interpreter.step();
                }

                if (interpreter.state() == Turing::Result::ACCEPT) {
                    cout << "Accepted!" << endl;
                    break;
                } else if (interpreter.state() == Turing::Result::REJECT) {
                    cout << "Rejected!" << endl;
                    break;
                } else if (interpreter.state() == Turing::Result::RUNNING) {
                    if (!getYesOrNo("Still running. Continue? ")) break;
                } else {
                    error("Huh?");
                }
            }
        }
    }
}

#include "GUI/MiniGUI.h"
#include "GVector.h"
#include "ProgramCore.h"
#include "Turing/Turing.h"
#include "gbrowserpane.h"
#include "goptionpane.h"
#include "gtimer.h"
#include "filelib.h"
#include "Utilities/Unicode.h"
#include "gthread.h"
#include <fstream>
#include <memory>
#include <unordered_map>
using namespace std;
using namespace MiniGUI;

namespace {
    const string kBackgroundColor = "white";
    const string kCodeBackgroundColor = "#c0c0c0";

    /* Default size for one character, as a fraction of the width of the screen. */
    const double kDefaultCharSize = 48.0 / 1000;

    const Font kCharFont(FontFamily::UNICODE_MONOSPACE, FontStyle::BOLD, 24, "black");
    const Font kCodeFont(FontFamily::UNICODE_MONOSPACE, FontStyle::NORMAL, 24, "black");

    const double kCodeLineHeight = 32;

    const string kCharBackgroundColor = "#ffffa6"; // Slide's color
    const string kCharBorderColor = "black";

    const string kActiveLineColor = "#c0ffc0";

    /* Number of characters to display at any one time, including the ellipses. */
    const int64_t kNumChars = 21;

    /* What happens on a seek to the end. */
    const size_t kStepsOnEndSeek  = 5000000;
    const string kContinueMessage = "We've run your program for a while and it's not done yet. Keep running it?";
    const string kContinueTitle   = "Program Still Running";

    /* Margin on each side of the tape that the tape head can't leave. If the tape
     * head's displayed position were to move outside of this range, instead we scroll
     * the tape over.
     */
    const int64_t kTapeHeadMargin = 3;

    /* Number of lines to keep below us when scrolling the current line number. */
    const size_t kLineMargin = 6;

    /* Minimum size of a scroll down before we just jump to the position. */
    const size_t kSmallJumpSize = 3;

    /* Vectors making up the "current position" arrow. Coordinates are relative
     * to the bounding box making up the arrow.
     */
    const vector<GVector> kArrow = {
        { 0.5,   0 },
        {   0, 0.5 },
        { 0.3, 0.5 },
        { 0.3,   1 },
        { 0.7,   1 },
        { 0.7, 0.5 },
        { 1.0, 0.5 }
    };
    const string kArrowColor = "black";

    /* Map from slider settings to speeds. */
    const map<int, double> kAnimationSpeeds = {
        {  0, 1250 },
        {  1, 1000 },
        {  2, 750  },
        {  3, 500  },
        {  4, 250  },
        {  5, 125  },
        {  6, 50   },
        {  7, 1    }
    };

    /* Aspect ratio. */
    const double kAspectRatio = 5.0 / 3.0;

    /* Fraction of the window height occupied by the program. */
    const double kProgramHeight = 0.85;

    /* Where characters go. All sizes are relative to the tape box. */
    const double kCharY           = 0.05 / kAspectRatio;
    const double kCharHeight      = 0.5 / kAspectRatio;

    /* Animation controls. */
    const double kAnimationSpeed = 10; //ms

    const string kWelcome = R"(Welcome to the TM Debugger!

    This tool lets you single-step through a Turing program to better understand how it works.

    Click "Load Program" to choose an automaton.)";
    const Font kWelcomeFont(FontFamily::SERIF, FontStyle::BOLD_ITALIC, 24, "#4C5866"); // Marengo

    const string kSingleErrorMessageTemplate = "%s.\n";
    const string kErrorMessageTemplate = "This automaton is not valid and therefore cannot be debugged. Please correct the following errors in the editor:\n%s";

    const Font kErrorFont(FontFamily::SERIF, FontStyle::NORMAL, 24, "#960018"); // Carmine

    class DebugGUI: public ProblemHandler {
    public:
        DebugGUI(GWindow& window);
        ~DebugGUI();

        void actionPerformed(GObservable* source) override;
        void changeOccurredIn(GObservable* source) override;
        void timerFired() override;
        void windowResized() override;

    protected:
        void repaint() override;

    private /* state */:
        shared_ptr<Turing::Program> program_ = make_shared<Turing::Program>(ifstream("res/Simple.tm", ios::binary));
        shared_ptr<Turing::Interpreter> interpreter_;

        /* Minimum character position to display. */
        int64_t lowIndex_ = -kTapeHeadMargin;

        /* Start and end points for each line of the program. */
        vector<pair<int, int>> lineBoundaries_;

        /* Minimum program line to display. */
        size_t lowLine_ = 0;
        size_t numLinesDisplayed_;

        /* Last input entered; used for the "rewind" feature. */
        string lastInput_;

        enum class UIState {
            INITIALIZING,
            CHOOSE_PROGRAM,
            CHOOSE_INPUT,
            RUNNING_PAUSE,
            RUNNING_PLAY
        };
        UIState state_ = UIState::INITIALIZING;

        /* Central display. */
        Temporary<GContainer> centerDisplay_;
          GBrowserPane* console_;
          GCanvas*      tapeArea_;

        /* Currently-selected panel. */
        GContainer* currentPanel_ = nullptr;

        Temporary<GContainer> mainPanel_;

        GContainer* emptyPanel_;

        GContainer* loadPanel_;
          GButton* loadButton_;
          GLabel*  currProgramLabel_;

        GContainer* debugPanel_;
          GButton* toBeginning_;
          GButton* toEnd_;
          GButton* step_;
          GButton* stop_;
          GButton* playPause_;
          GSlider* speedControl_;

        GContainer* inputPanel_;

        GTimer* timer_ = new GTimer(kAnimationSpeeds.at(0)); // 07/08/21 Have to leak this due to bugs in StanfordCPPLib

        GContainer* activePanel_ = nullptr;
          GTextField* inputField_ = nullptr;
          GButton*    startButton_;

    private /* helpers */:
        GRectangle programArea() const;
        GRectangle tapeArea() const;

        void drawProgramArea();
        void drawTapeArea();

        GRectangle worldToGraphics(const GRectangle& in) const;
        void drawSingleCharacter(char32_t ch, const GRectangle& worldBounds);
        void drawArrow(const GRectangle& graphicsBounds);

        void setState(UIState state);
        void setPanel(GContainer* panel);

        void initSimulation(const string& input);
        void step();
        void seekToEnd();

        void resetInputPanel();

        void userLoadProgram();
        void loadProgram(const string& filename);

        void recalcDisplayedLines();
        void recenterLine();
        void jumpToLine(size_t lineNo);

        void computeText();
    };

    DebugGUI::DebugGUI(GWindow& window) : ProblemHandler(window) {
        mainPanel_ = make_temporary<GContainer>(window, "SOUTH", GContainer::LAYOUT_GRID);
          loadPanel_ = new GContainer(GContainer::LAYOUT_FLOW_VERTICAL);
            currProgramLabel_ = new GLabel("Choose a Program");
            loadButton_ = new GButton("Load Program");
            loadPanel_->add(currProgramLabel_);
            loadPanel_->add(loadButton_);
          mainPanel_->addToGrid(loadPanel_, 0, 0);

          currentPanel_ = new GContainer();
          mainPanel_->addToGrid(currentPanel_, 0, 1, 1, 3);

        emptyPanel_ = new GContainer();

        debugPanel_ = new GContainer(GContainer::LAYOUT_FLOW_VERTICAL);
          GContainer* buttons = new GContainer();
            toBeginning_ = new GButton("⏮");
            playPause_ = new GButton("▶");
            step_ = new GButton("⏩");
            toEnd_ = new GButton("⏭️");
            stop_ = new GButton("⏹");
            buttons->add(toBeginning_);
            buttons->add(playPause_);
            buttons->add(step_);
            buttons->add(toEnd_);
            buttons->add(stop_);
          debugPanel_->add(buttons);
          GContainer* speed = new GContainer();
            speed->add(new GLabel("Speed: "));
            speedControl_ = new GSlider(0, kAnimationSpeeds.size() - 1, 0);
            speed->add(speedControl_);
          debugPanel_->add(speed);

        inputPanel_ = new GContainer();
        resetInputPanel();

        /* Bug: Non-added panels must be marked invisible. */
        debugPanel_->setVisible(false);
        inputPanel_->setVisible(false);

        centerDisplay_ = make_temporary<GContainer>(window, "CENTER", GContainer::LAYOUT_FLOW_VERTICAL);
          console_ = new GBrowserPane();
          tapeArea_ = new GCanvas();
          tapeArea_->setAutoRepaint(false);

          centerDisplay_->add(console_);
          centerDisplay_->add(tapeArea_);

          console_->setSize(window.getCanvasWidth() * 0.95, window.getCanvasHeight() * 0.95 * kProgramHeight);
          tapeArea_->setSize(window.getCanvasWidth() * 0.95, window.getCanvasHeight() * 0.95 * (1 - kProgramHeight));

        setState(UIState::CHOOSE_PROGRAM);
        mainPanel_->setWidth(window.getWidth() * 0.95);
    }

    DebugGUI::~DebugGUI() {
        timer_->stop();
    }

    void DebugGUI::actionPerformed(GObservable* source) {
        /* The "load" button always works. */
        if (source == loadButton_) {
            userLoadProgram();
        } else if (state_ == UIState::CHOOSE_INPUT) {
            if (source == startButton_) {
                initSimulation(inputField_->getText());
            }
        } else if (state_ == UIState::RUNNING_PAUSE || state_ == UIState::RUNNING_PLAY) {
            if (source == step_) {
                setState(UIState::RUNNING_PAUSE);
                step();
            } else if (source == playPause_) {
                setState(state_ == UIState::RUNNING_PAUSE? UIState::RUNNING_PLAY : UIState::RUNNING_PAUSE);
            } else if (source == stop_) {
                setState(UIState::CHOOSE_INPUT);
            } else if (source == toBeginning_) {
                initSimulation(lastInput_);
            } else if (source == toEnd_) {
                setState(UIState::RUNNING_PAUSE);
                seekToEnd();
            }
        }
    }

    void DebugGUI::initSimulation(const string& strInput) {
        /* Build the input string. */
        vector<char32_t> input;
        for (char32_t ch: utf8Reader(strInput)) {
            input.push_back(ch);
        }

        lastInput_ = strInput;
        interpreter_ = make_shared<Turing::Interpreter>(*program_, input);

        /* One buffer cell so going before the start of the tape doesn't shift. */
        lowIndex_ = -kTapeHeadMargin - 1;
        lowLine_ = 0;
        recalcDisplayedLines();

        setState(UIState::RUNNING_PAUSE);
        recalcDisplayedLines();
        requestRepaint();
    }

    GRectangle DebugGUI::programArea() const {
        return { 0, 0, window().getWidth(), window().getCanvasHeight() * kProgramHeight };
    }
    GRectangle DebugGUI::tapeArea() const {
        return { 0, 0, tapeArea_->getWidth(), tapeArea_->getHeight() };
    }

    void DebugGUI::repaint() {
        clearDisplay(window(), kBackgroundColor);

        switch (state_) {
            case UIState::CHOOSE_INPUT:
            case UIState::RUNNING_PAUSE:
            case UIState::RUNNING_PLAY:
                drawProgramArea();
                drawTapeArea();
                break;

            case UIState::CHOOSE_PROGRAM:
                // TODO: This!
                break;

            default:
                break;
        }
    }

    void DebugGUI::changeOccurredIn(GObservable* source) {
        if (source == speedControl_) {
            timer_->setDelay(kAnimationSpeeds.at(speedControl_->getValue()));
            cout << "Speed is now " << timer_->getDelay() << endl;
        }
    }

    void DebugGUI::drawProgramArea() {
        #if 0
            window().setColor(kCodeBackgroundColor);
            window().fillRect(programArea());

            /* How many lines can we fit? */
            auto bounds = programArea();

            for (size_t i = lowLine_; i < lowLine_ + numLinesDisplayed_; i++) {
                /* Skip anything past the end. */
                if (i >= program_->numLines()) continue;

                GRectangle lineBounds = {
                    bounds.x, bounds.y + (i - lowLine_) * kCodeLineHeight,
                    bounds.width, kCodeLineHeight
                };

                /* Highlight the current line. */
                if (interpreter_ && i == interpreter_->lineNumber()) {
                    window().setColor(kActiveLineColor);
                    window().fillRect(lineBounds);
                }

                GRectangle lineFull = lineBounds;
                lineFull.width = numeric_limits<double>::infinity();

                auto text = TextRender::construct(program_->line(i), lineFull, kCodeFont, LineBreak::NO_BREAK_SPACES);
                text->alignCenterVertically();
                text->draw(window());
            }
        #endif
    }

    void DebugGUI::drawTapeArea() {
        GRectangle bounds = tapeArea();

        /* Wipe anything in the tape area that may have overdrawn. */
        // TODO: Still necessary?
        tapeArea_->setColor(kBackgroundColor);
        tapeArea_->fillRect(bounds);

        /* No interpreter? Nothing to draw. */
        if (!interpreter_) return;

        /* Determine the width to use, done in graphics coordinates. We're chosen so that
         *
         * (1) we don't end up with total width greater than the screen size, and
         * (2) we don't exceed the height of the character area.
         *
         * The calculation in (1) adds one to the total to account for the arrow.
         */
        double width = bounds.width / double(kNumChars + 1);
        width = min(width, bounds.height / 2 - 2 * bounds.height * kCharY);

        double baseX = bounds.x + (bounds.width - width * kNumChars) / 2.0;
        double baseY = bounds.y;

        /* Draw each character. */
        for (size_t i = 0; i < kNumChars; i++) {
            char32_t ch = (i == 0 || i == kNumChars - 1)? fromUTF8("⋯") : interpreter_->tapeAt(i + lowIndex_);
            drawSingleCharacter(ch, { baseX + width * i, baseY + kCharY * bounds.height, width, width });
        }

        /* Draw the arrow at the current position. */
        int64_t offset = interpreter_->tapeHeadPos() - lowIndex_;
        drawArrow({ baseX - width / 2 + width * (0.5 + offset), baseY + bounds.height * kCharY + width, width, width });

        tapeArea_->repaint();
    }

    void DebugGUI::drawSingleCharacter(char32_t ch, const GRectangle& bounds) {
        /* Draw the bounding box. */
        tapeArea_->setColor(kCharBackgroundColor);
        tapeArea_->fillRect(bounds);
        tapeArea_->setColor(kCharBorderColor);
        tapeArea_->drawRect(bounds);

        /* Draw the text. */
        auto text = TextRender::construct(toUTF8(ch), bounds, kCharFont);
        text->alignCenterVertically();
        text->alignCenterHorizontally();
        text->draw(tapeArea_);
    }

    void DebugGUI::drawArrow(const GRectangle& bounds) {
        GPolygon polygon;
        polygon.setFilled(true);
        polygon.setColor(kArrowColor);

        /* Box origin point. */
        GPoint box = { bounds.x, bounds.y };

        /* Box coordinate transform:
         *  | width    0   |
         *  |   0   height |
         */
        GMatrix transform = { bounds.width, 0, 0, bounds.height };
        for (const auto& v: kArrow) {
            polygon.addVertex(box + transform * v);
        }

        tapeArea_->draw(&polygon);
    }

    namespace {
        string anchorFor(int line) {
            return "<a name=\"" + to_string(line) + "";
        }
    }

    void DebugGUI::step() {
        interpreter_->step();

        /* Constrain the tape head position. */
        if (interpreter_->tapeHeadPos() - lowIndex_ < kTapeHeadMargin) {
            lowIndex_--;
        } else if (interpreter_->tapeHeadPos() - lowIndex_ > kNumChars - kTapeHeadMargin) {
            lowIndex_++;
        }

        /* Jump to the current line. */
        //auto range = lineBoundaries_[interpreter_->lineNumber()];
        //console_->select(range.first, range.second);
        auto qTextEdit = dynamic_cast<QTextEdit*>(console_->getWidget());
        if (!qTextEdit) error("Can't get underlying widget.");
        qTextEdit->find(QString::fromStdString(anchorFor(interpreter_->lineNumber()) + program_->line(interpreter_->lineNumber())));


        recenterLine();
        requestRepaint();
    }

    void DebugGUI::setState(UIState state) {
        if (state == state_) return; // Nothing to do

        /* If we're currently in the "play" state, stop the timer. */
        if (state_ == UIState::RUNNING_PLAY) {
            timer_->stop();
        }

        if (state == UIState::CHOOSE_PROGRAM) {
            setPanel(emptyPanel_);
        } if (state == UIState::CHOOSE_INPUT) {
            setPanel(inputPanel_);
        } else if (state == UIState::RUNNING_PAUSE) {
            playPause_->setText("▶");

            setPanel(debugPanel_);
        } else if (state == UIState::RUNNING_PLAY) {
            playPause_->setText("⏸");

            /* If switching to the "play" state, engage the timer. */
            setPanel(debugPanel_);
            timer_->start();
        }

        state_ = state;

        /* Repaint; we may have a totally different look now! */
        requestRepaint();
    }

    void DebugGUI::timerFired() {
        if (state_ == UIState::RUNNING_PLAY) {
            step();
            if (interpreter_->state() != Turing::Result::RUNNING) {
                setState(UIState::RUNNING_PAUSE);
            }
        }
    }

    void DebugGUI::setPanel(GContainer* panel) {
        GThread::runOnQtGuiThread([&] {
            /* Already installed? Nothing to do. */
            if (activePanel_ == panel) return;

            if (activePanel_) {
                currentPanel_->remove(activePanel_);
                activePanel_->setVisible(false);
            }

            /* Handle bug with the input panel. */
            if (panel == inputPanel_) {
                resetInputPanel();
                panel = inputPanel_;
            }

            /* Set this as the active panel. */
            currentPanel_->add(panel);
            activePanel_ = panel;
            panel->setVisible(true);

            /* Request a repaint; the content height may have changed. */
            requestRepaint();
        });
    }

    /* There appears to be an internal bug where removing an input
     * element and adding it triggers a Qt internal warning. Destroy and
     * then restore the input field.
     */
    void DebugGUI::resetInputPanel() {
        /* Recycle? */
        string contents;
        if (inputField_) {
            contents = inputField_->getText();
            inputPanel_->clear();
            delete inputField_;
        }
        /* Fresh? */
        else {
            startButton_ = new GButton("Debug");
        }
        inputField_ = new GTextField(contents);
        inputField_->setPlaceholder("ε");

        inputPanel_->add(new GLabel("Input: "));
        inputPanel_->add(inputField_);
        inputPanel_->add(startButton_);
    }

    void DebugGUI::userLoadProgram() {
        /* Ask user to pick a file; don't do anything if they don't pick one. */
        string filename = GFileChooser::showOpenDialog(&window(), "Choose Program", "res/", "*.tm");
        if (filename == "") return;

        loadProgram(filename);
    }

    void DebugGUI::loadProgram(const string& filename) {
        ifstream input(filename, ios::binary);
        if (!input) error("Could not open the file " + filename);

        program_ = make_shared<Turing::Program>(input);
        interpreter_ = nullptr;
        currProgramLabel_->setText(getTail(filename));

        computeText();

        setState(UIState::CHOOSE_INPUT);
        recalcDisplayedLines();
        requestRepaint();
    }

    void DebugGUI::seekToEnd() {
        setDemoOptionsEnabled(false);
        mainPanel_->setEnabled(false);
        do {
            for (size_t i = 0; i < kStepsOnEndSeek && interpreter_->state() == Turing::Result::RUNNING; i++) {
                step();
            }
        } while(interpreter_->state() == Turing::Result::RUNNING &&
                GOptionPane::showConfirmDialog(window().getWidget(), kContinueMessage, kContinueTitle) == GOptionPane::CONFIRM_YES);
        mainPanel_->setEnabled(true);
        setDemoOptionsEnabled(true);
        requestRepaint();
    }

    void DebugGUI::jumpToLine(size_t lineNo) {
        /* Put a few lines above us for some context. */
        if (lineNo < numLinesDisplayed_ / 4) {
            lowLine_ = 0;
        } else {
            lowLine_ = lineNo - numLinesDisplayed_ / 4;
        }
    }

    void DebugGUI::recenterLine() {
        if (!program_) return;
        size_t lineNo = interpreter_? interpreter_->lineNumber() : 0;

        /* Scroll up if we need to. */
        if (lineNo < lowLine_) {
            jumpToLine(lineNo);
        }
        /* Scroll down if we need to. */
        else if (lineNo + kLineMargin > lowLine_ + numLinesDisplayed_) {
            /* See how big a jump this is. */
            size_t delta = lineNo + kLineMargin - lowLine_ - numLinesDisplayed_;

            if (delta < kSmallJumpSize) {
                lowLine_ += delta;
            } else {
                jumpToLine(lineNo);
            }
        }

        /* And, as always, ensure we don't scroll past the end. */
        if (lowLine_ + numLinesDisplayed_ > program_->numLines()) {
            if (numLinesDisplayed_ >= program_->numLines()) {
                lowLine_ = 0;
            } else {
                lowLine_ = program_->numLines() - numLinesDisplayed_;
            }
        }
    }

    void DebugGUI::recalcDisplayedLines() {
        numLinesDisplayed_ = ceil(programArea().height / kCodeLineHeight);
        recenterLine();
    }

    void DebugGUI::windowResized() {
        recalcDisplayedLines();
        ProblemHandler::windowResized();
    }

    void DebugGUI::computeText() {
        lineBoundaries_.clear();

        ostringstream contents;
        contents << R"(<html><head></head><body><pre style="font-size:24pt;">)";

        for (size_t i = 0; i < program_->numLines(); i++) {
            contents << anchorFor(i) << program_->line(i) << endl;
        }

        contents << anchorFor(program_->numLines()) << endl;

        contents << "</pre></body></html>";

        console_->setText(contents.str());
    }
}

GRAPHICS_HANDLER("Turing Debugger", GWindow& window) {
    return make_shared<DebugGUI>(window);
}
